﻿/*
	VERSION:		1.8
	
	DESCRIPTION:
		This is a movieClip-based music system.
		It simulates stereo using mono music clips.
	
	USAGE:
		#include "makeStereoMusic.as"
		MUSIC_MC = makeStereoMusic();
		MUSIC_MC.play( "songName" );
		
		makeStereoMusic( target_mc:MovieClip, newName:String, [newDepth:Number] );
		
	FUNCTIONS:
		play( "songName" );				// Plays the songName_intro tune, then switches to songName (which should loop)
		playLoop( "songName" );		// Skip any intro's and specifically play the main loo pof the song
		stopMusic();						// Stops the music and forgets the last song
		fadeOut( seconds );				// Fades out, then stops the music
		
	PROPERTIES:
		volume								// scalar (percentage) that gets or sets the maximum playback volume of all music
		innerVolume							// gets or sets the intensity of THIS music  (used for fade-out's)
		
		Playback volume  =  innerVolume * (volume/100)
		To change the volume of all music played, use "volume".
		To change the volume of only the current song, use "innerVolume".
		All songs start playback with an "innerVolume" of 100.
		
	INTROS & LOOPS:
		You may add folders before the fileNames.
		Name your files / linkages like so:
			songName_intro.mp3			play("songName.mp3")				(The extension must be .mp3)
			songName.mp3					playLoop("songName.mp3")			(This is the looping version of the song.)
		When play() is used, it'll automatically attempt to load an intro for the song. If there's no intro, it'll play the loop.
*/



makeStereoMusic = function( target_mc, newName, newDepth )
{
	// resolve optional parameters
	var target_mc = (target_mc != undefined) ? target_mc : this;
	var newName = (newName != undefined) ? newName : "musicSystem_mc";
	#include "nextDepth.as"
	var newDepth = (newDepth) ? newDepth : nextDepth(target_mc);
	
	// create containers
	var _this = target_mc.createEmptyMovieClip( newName, newDepth );
	_this.createEmptyMovieClip( "right_mc", 1 );			// container for rightSound's transforms
	_this.createEmptyMovieClip( "left_mc", 0 );			// container for leftSound's transforms
	_this.rightSound = new Sound( _this.right_mc );	// internal Sound object
	_this.leftSound = new Sound( _this.left_mc );		// internal Sound object
	
	// internal variables
	_this.songName = "";										// Linkage name		(remember which sound is being played, for stop()  &  play()
	_this.loops = 999999;									// loops indefinitely
	_this.stereoDelay = 0.02;								// 20 milliseconds
	_this.volume = 100;										// 100%  (for fadeOut()
	_this.isPlaying = false;
	// Mimic Sound events		(these can be externally defined)
	_this.onID3 = function(){}
	_this.onLoad = function(){}
	_this.onSoundComplete = function(){}
	
	
	
	
	
	
	// FUNCTIONS
	_this.play = function( songName )
	{
		if(_this.fade)
		{
			_this.fade.stop();
			delete _this.fade;
			_this.songName = "";
		}// if:  fading-out
		_this.setInnerVolume(100);
		
		if(_this.songName != songName)
		{
			// transition:  intro  ->  loop
			_this.rightSound.onSoundComplete = function()
			{
				// play normally
				var songName = _this.songName;
				_this.songName = "";					// spoof it so it'll play the same song again
				_this.playLoop( songName );		// play the regular (looping) version
			}// onSoundComplete()
			
			_this.rightSound.stop();
			_this.leftSound.stop();
			var nameLength = songName.indexOf(".");
			var introName = songName.substr(0, nameLength) + "_intro" + ".mp3";		// songName.mp3  ->  songName_intro.mp3
			_this.songName = songName;
			_this.isPlaying = false;
			_this.rightSound.attachSound( introName );
			_this.leftSound.attachSound( introName );
			
			// If there is no intro,  immediately play the loop
			if(_this.rightSound.duration == undefined)		// if there is no intro
				_this.rightSound.onSoundComplete();		// immediately play looping version
		}// if: a new song is specified
		
		if( _this.isPlaying == false )
		{
			// Stop the sound
			_this.rightSound.stop();				// probably redundant
			_this.leftSound.stop();				// probably stops ALL instances of this particular sound effect
			// start right sound, 20 milliseconds sooner
			_this.rightSound.start( 0+_this.stereoDelay, 1 );
			// start left sound
			_this.leftSound.start( 0, 1 );
			_this.setChannels();
			_this.isPlaying = true;
		}// if:  not playing
	}// play()
	
	
	
	_this.playLoop = function( songName )
	{
		if(_this.songName != songName)
		{
			_this.rightSound.stop();
			_this.leftSound.stop();
			_this.rightSound.attachSound( songName );
			_this.leftSound.attachSound( songName );
			_this.songName = songName;
			_this.isPlaying = false;
			
			// Remove intro  ->  loop transition
			_this.rightSound.onSoundComplete = function(){}
		}// if: a new song is specified
		
		if( _this.isPlaying == false )
		{
			// Stop the sound
			_this.rightSound.stop();			// probably redundant
			_this.leftSound.stop();				// probably stops ALL instances of this particular sound effect
			// start right sound, 20 milliseconds sooner
			_this.rightSound.start( 0+_this.stereoDelay, _this.loops );
			// start left sound
			_this.leftSound.start( 0, _this.loops );
			_this.setChannels();
			_this.isPlaying = true;
		}// if:  not playing
	}// playLoop()
	
	
	
	_this.stop = function()
	{
		// Stop the sound
		_this.rightSound.stop();			// probably redundant
		_this.leftSound.stop();				// probably stops ALL instances of this particular sound effect
		_this.isPlaying = false;
		_this.songName = "";					// Forget the last song. Next time, it'll play again from the beginning.
	}// stop()
	
	
	
	_this.stopMusic = _this.stop;		// compatible with my simpler music system
	
	
	
	// setVolume()
	_this.setVolume = function( newVolume )
	{
		_this.rightSound.setVolume( newVolume );
		_this.leftSound.setVolume( newVolume );
		_this.volume = newVolume;
	}// volume		(set)
	// getVolume()
	_this.getVolume = function()
	{
		return _this.volume;
	}// volume		(get)
	_this.addProperty( "volume", _this.getVolume, _this.setVolume );
	
	
	
	_this.getInnerVolume = function()
	{
		return _this.rightSound.getVolume() / (_this.volume/100);		// song volume  (global volume factored out)
	}// volume		(get)
	_this.setInnerVolume = function( newVolume )
	{
		newVolume *= (_this.volume/100);		// innerVolume * global music volume
		_this.rightSound.setVolume( newVolume );
		_this.leftSound.setVolume( newVolume );
	}// volume		(set)
	_this.addProperty( "innerVolume", _this.getInnerVolume, _this.setInnerVolume );
	
	
	
	_this.getDuration = function()
	{
		return _this.leftSound.duration;
	}// duration	(get)
	_this.addProperty( "duration", _this.getDuration, null );
	
	
	
	_this.fadeOut = function( seconds )
	{
		if(_this.volume > 0  &&
		   _this.fade == undefined)
		{
			// tween volume from current to ZERO
			var startVolume = _this.getInnerVolume();
			_this.fade = new mx.transitions.Tween( _this, "innerVolume", null, startVolume, 0, seconds, true);
			
			// onChange
			_this.fade.onMotionChanged = function()
			{
				_this.setInnerVolume( this.position );
			}// onMotionChanged()
			
			// onDone
			_this.fade.onMotionFinished = function()
			{
				delete _this.fade;
				_this.stop();
			}// onMotionFinished()
		}// if:  volume isn't ZERO
	}// fadeOut()
	
	
	
	// Set channels		(which speaker each sound object uses)
	_this.setChannels = function()
	{
		_this.rightSound.setPan( 100 );
		_this.leftSound.setPan( -100 );
	}// setChannels()
	
	
	
	// pass Sound events
	_this.passSoundEvents = function()
	{
		_this.leftSound.onID3 = function()
		{
			_this.broadcastMessage( "onID3" );
			_this.onID3();
		}
		_this.leftSound.onLoad = function( success )
		{
			_this.broadcastMessage( "onLoad" );
			_this.onLoad( success );
		}
		_this.leftSound.onSoundComplete = function()
		{
			_this.broadcastMessage( "onSoundComplete" );
			_this.onSoundComplete();
		}
	}// passSoundEvents()
	
	
	
	
	
	
	// SETUP
	AsBroadcaster.initialize( this );
	// Set channels				(which speaker each sound object uses)
	_this.setChannels();
	// pass sound events		(allows addListener(), which is not available in normal Sound objects)
	_this.passSoundEvents();
	
	// return this sound system
	return _this;
}// makeStereoMusic()